function [C, H] = contmix (func, levels, region, lines, names, delta)
% contmix -- Contourplot for 3-mixture-functions
%
% 	$Id: contmix.m,v 1.9 2005/12/06 13:12:26 bhm Exp $	
%
% [C, H] = contmix (func, levels, region, lines, names, delta)
%
% Output:
% C:    Matrix as described in CONTOURC
% H:    A column vector of handles to LINE or PATCH objects, one handle per line.
%       Both of these can be used as input to CLABEL.
%
% Input:
% func: Name of a function (enclosed in "'"s).  The function must be a
%       function of three mixture variables, and must accept vectors
%       as arguments, i.e. func(a, b, c) with a, b and c vectors of
%       equal length. 
% levels:  Optional.  A vector with the levels to use for the
%       contours, or an integer giving the desired number of levels.  If
%       not present, MATLAB's automatic levels are used.
% region:  Optional.  A 3 by 2 matrix where each row gives the lower
%       and upper limits of the corresponding component.  The matrix
%       defines the region of interest, in mixture coordinates.  The plot
%       will be restricted to this region.  The default is to use the whole
%       triangle.
% lines:  Optional.  A cell array specifying which support lines are to be
%       plotted:
%       'boundary' means the boundary of the region,
%       'limits' means the lines representing the upper and lower limits
%                (adjusted so that they are all active),
%       'inactive' means the inactive (original) limits,
%       'triangle' means the triangle surrounding the mixture coordinate
%                system.
%       The lines are plotted in the order listed in lines.  Default is
%       {'boundary', 'triangle'}.  Note that the "{" and "}" are
%       compulsory, even when specifying only a single line type.
%       Specifying an empty cell array ("{ }") gives no lines at all.
% names: Optional.  A cell array with the names of the components, to
%       label the vertices of the triangle.
% delta:  Optional.  The mesh size to use in building the grid.  Default 1/50.
%
% Copyright 2004 Bjrn-Helge Mevik.  Email: bjorn-helge.mevik(a)matforsk.no

% This program is free software; you can redistribute it and/or modify 
% it under the terms of the GNU General Public License version 2 as
% published by the Free Software Foundation.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

if exist (func) ~= 2  % Not a function file
  error ('"func" must be the name of a function file.')
end

% The default grid size:
if nargin < 6 | isempty (delta)
  delta=1/50;
end

%% Note: All calculations are performed in mixture coordinates, with
%% either the first two coordinates or all three.  Plotting is performed
%% in independent coordinates; see the explanation of the matrix M below.

%% Evaluate the function over a grid in the two first mixture coordinates:
a = 0:delta:1;
b = a;
[a_grid,b_grid] = meshgrid(a,b);
a_vec = a_grid(:);
b_vec = b_grid(:);
resp_vec = feval (func, a_vec, b_vec, 1 - a_vec - b_vec);
resp_grid = reshape(resp_vec, size(a_grid));

%% If region is present; replace values outside the given region with NaN's
if nargin > 2 & ~isempty (region)
  % Sanity checks:
  if min (region(:)) < 0 | max (region(:)) > 1
     error ('Invalid "region": limits outside the mixture coordinate system.')
  end
  if ~all (size (region) == [3 2])
    error ('"region" must be a matrix with three rows and two coloumns.')
  end
  % Change any passive constraints (and give a warning)
  tmp = zeros (size (region));
  tmp(1,1) = max (region(1,1), 1 - region(2,2) - region(3,2));
  tmp(2,1) = max (region(2,1), 1 - region(1,2) - region(3,2));
  tmp(3,1) = max (region(3,1), 1 - region(1,2) - region(2,2));
  tmp(1,2) = min (region(1,2), 1 - region(2,1) - region(3,1));
  tmp(2,2) = min (region(2,2), 1 - region(1,1) - region(3,1));
  tmp(3,2) = min (region(3,2), 1 - region(1,1) - region(2,1));
  region_old = region;			% Save the old limits as region_old
  region = tmp;				% The new region limits
  if any (region(:) ~= region_old(:))
    warning ('Inactive constraints detected (and adjusted).')
  end
  resp_grid(a_grid < region(1,1)) = NaN;  % Lower limit, a
  resp_grid(a_grid > region(1,2)) = NaN;  % Upper limit, a
  resp_grid(b_grid < region(2,1)) = NaN;  % Lower limit, b
  resp_grid(b_grid > region(2,2)) = NaN;  % Upper limit, b
  % The delta/2 is added to avoid cutting contours too short.  Not robust.
  resp_grid(1 - a_grid - b_grid < region(3,1) - delta/2) = NaN;% Lower limit, c
  resp_grid(1 - a_grid - b_grid > region(3,2) + delta/2) = NaN;% Upper limit, c
end
% In all cases, make sure values outside the triangle (c < 0) are NaN:
resp_grid(1 - a_grid - b_grid < 0 - delta/2) = NaN;

%% The conversion matrix (it converts from mixture to independent coordinates):
%% ([a b c] - 1/3) * M = [x y 0].  This gives
%% (1/3,1/3,1/3) <=> ( 0,          0)
%% (1,0,0)       <=> (-1/sqrt(2), -1/sqrt(6))
%% (0,1,0)       <=> ( 1/sqrt(2), -1/sqrt(6))
%% (0,0,1)       <=> ( 0,          sqrt(2/3))
%% Thus the x-axis is the difference between b and a, and the y-axis is c.
M = [ -sqrt(3), -1, sqrt(2)
  sqrt(3), -1, sqrt(2)
  0, 2, sqrt(2)] / sqrt(6);

%% Default support lines:
if nargin < 4 | isempty (lines)
  if nargin < 3 | isempty (region)
    lines = {'triangle'};
  else
    lines = {'boundary', 'triangle'};
  end
end

%% Plot the support lines:
for i = 1:length(lines)
  switch lines{i}
   case 'boundary'
    la = region (1,1); ua = region (1,2); % Short names for the limits
    lb = region (2,1); ub = region (2,2);
    lc = region (3,1); uc = region (3,2);
    tmp = [ ua      1-ua-lc lc
	    1-ub-lc ub      lc
	    la      ub      1-la-ub
	    la      1-la-uc uc
	    1-lb-uc lb      uc
	    ua      lb      1-ua-lb
	    ua      1-ua-lc lc ];
    tmp = (tmp - 1/3) * M;
    plot (tmp(:,1), tmp(:,2))
    hold on;
   case 'limits'
    la = region (1,1); ua = region (1,2); % Short names for the limits
    lb = region (2,1); ub = region (2,2);
    lc = region (3,1); uc = region (3,2);
    tmp = [ la   0    1-la
	    ua   0    1-ua
	    0    lb   1-lb
	    0    ub   1-ub
	    0    1-lc lc
	    0    1-uc uc
	    la   1-la 0
	    ua   1-ua 0
	    1-lb lb   0
	    1-ub ub   0
	    1-lc 0    lc
	    1-uc 0    uc ];
    tmp = (tmp - 1/3) * M;
    plot ([tmp(1:6,1), tmp(7:12,1)]', [tmp(1:6,2), tmp(7:12,2)]', 'k')
    hold on;
   case 'inactive'
    la = region_old (1,1); ua = region_old (1,2); % Short names for the limits
    lb = region_old (2,1); ub = region_old (2,2);
    lc = region_old (3,1); uc = region_old (3,2);
    tmp = [ la   0    1-la
	    ua   0    1-ua
	    0    lb   1-lb
	    0    ub   1-ub
	    0    1-lc lc
	    0    1-uc uc
	    la   1-la 0
	    ua   1-ua 0
	    1-lb lb   0
	    1-ub ub   0
	    1-lc 0    lc
	    1-uc 0    uc ];
    tmp = (tmp - 1/3) * M;
    plot ([tmp(1:6,1), tmp(7:12,1)]', [tmp(1:6,2), tmp(7:12,2)]', 'k--')
    hold on;
   case 'triangle'
    tmp = ([1,0,0; 0,1,0; 0,0,1; 1,0,0] - 1/3) * M;
    plot (tmp(:,1), tmp(:,2))
    hold on;
   otherwise
    warning (['Unknown support line type: '''  lines{i}  ''''])
  end
end

%% Plot the contours in the first two mixture coordinates
%% FIXME: Quick fix: Matlab changed the return values of contour in ver 7.
if str2num(version('-release')) < 14
  if nargin > 1 & ~isempty(levels)
    [C,H] = contour (a, b, resp_grid, levels);
  else
    [C,H] = contour (a, b, resp_grid);
  end
else
  if nargin > 1 & ~isempty(levels)
    [C,H] = contour ('v6', a, b, resp_grid, levels);
  else
    [C,H] = contour ('v6', a, b, resp_grid);
  end
end  

%% Traverse throught the contour line matrix and convert the coordinates
%% of each contour line to mixture coordinates.  I have to transform both
%% the coordinates in C and in H, because later CLABEL calls needs C.
ind=1;
for i = 1:length (H)
  %% First update H
  % Extract the A and B coordinates (including final NaN):
  l1 = get (H(i), 'Xdata');
  l2 = get (H(i), 'Ydata');
  % Calculate the C coordinate:
  l3 = 1 - l1 - l2;
  % Transform these to cartesian coordinates:
  l = ([l1 l2 l3] - 1/3) * M;
  % Update:
  set(H(i),'Xdata',l(:,1));
  set(H(i),'Ydata',l(:,2));
  %% Then update C
  % The length of the current contour line: 
  len = C(2,ind);
  % Extract the A and B coordinates
  l1 = C(1,ind + (1:len));
  l2 = C(2,ind + (1:len));
  % Calculate the C coordinate:
  l3 = 1 - l1 - l2;
  % Transform these to cartesian coordinates:
  l = ([l1; l2; l3]' - 1/3) * M;
  % Update:
  C(1,ind + (1:len)) = l(:,1)';
  C(2,ind + (1:len)) = l(:,2)';
  % Jump to next contour line:
  ind = ind + len + 1;
end
axis equal
axis off

% Plot names, if given:
if nargin > 4 & ~isempty (names)
  text ([-0.75, 0.75, 0], [-0.47, -0.47, 0.87], names, 'HorizontalAlignment', 'Center')
end

hold off
